<?php
/*
 * ----------------------------------------------------------------------
 *
 *                          Borlabs Cookie
 *                      developed by Borlabs
 *
 * ----------------------------------------------------------------------
 *
 * Copyright 2018 Borlabs - Benjamin A. Bornschein. All rights reserved.
 * This file may not be redistributed in whole or significant part.
 * Content of this file is protected by international copyright laws.
 *
 * ----------------- Borlabs Cookie IS NOT FREE SOFTWARE -----------------
 *
 * @copyright Borlabs - Benjamin A. Bornschein, https://borlabs.io
 * @author Benjamin A. Bornschein, Borlabs ben@borlabs.io
 *
 */

namespace BorlabsCookie\Cookie;

use BorlabsCookie\Cookie\Backend\BlockedContentTypes;
use BorlabsCookie\Cookie\Frontend\ContentBlocker;
use BorlabsCookie\Cookie\Frontend\JavaScript;

class ThirdPartyHelper
{

    private static $instance;

    public static function getInstance()
    {

        if (null === self::$instance) {
            self::$instance = new self;
        }

        return self::$instance;
    }

    private function __clone()
    {
    }

    private function __wakeup()
    {
    }

    protected function __construct()
    {
    }

    /**
     * addBlockedContentType function.
     *
     * This functions allows you to add a new Blocked Content Type (BCT) to Borlabs Cookie.
     * The function will add your BCT in every language, if the site is using a multilanguage plugin.
     * The function also ensures that your BCT will only be added when it does not exist.
     *
     * @access public
     * @param string $typeId: only lowercase
     * @param string $name: title of your BCT
     * @param string $description: optional description
     * @param array $hosts: hosts for autodetection (for iframes or oEmbeds)
     * @param string $previewHTML: html code that is shown instead of the blocked content
     * @param string $globalJS: global JavaScript, loaded once
     * @param string $initJS: initialization JavaScript, executed everytime when a blocked content gets unblocked
     * @param array $settings (default: [])
     * @param bool $status (default: false)
     * @param bool $undeleteable (default: false): if true the user can not delete your BCT
     * @return void
     */
    public function addBlockedContentType($typeId, $name, $description = '', $hosts, $previewHTML, $globalJS = '', $initJS = '', $settings = [], $status = false, $undeleteable = false)
    {
        if (preg_match('/^[a-z]{3,}$/', $typeId)) {
            if (defined('ICL_LANGUAGE_CODE')) {

                $activeLanguages = apply_filters('wpml_active_languages', null, []);

                foreach ($activeLanguages as $languageData) {
                    if (!empty($languageData['code'])) {
                        BlockedContentTypes::getInstance()->add(
                            $typeId,
                            $name,
                            $description,
                            $hosts,
                            $previewHTML,
                            $globalJS,
                            $initJS,
                            $settings,
                            $status,
                            $undeleteable,
                            $languageData['code']
                        );
                    }
                }

            } else {
                BlockedContentTypes::getInstance()->add(
                    $typeId,
                    $name,
                    $description,
                    $hosts,
                    $previewHTML,
                    $globalJS,
                    $initJS,
                    $settings,
                    $status,
                    $undeleteable
                );
            }

            return true;
        } else {
            return false;
        }
    }

    /**
     * deleteBlockedContentType function.
     *
     * Deletes a Blocked Content Type by its typeId.
     *
     * @access public
     * @param mixed $typeId
     * @return void
     */
    public function deleteBlockedContentType($typeId)
    {
        global $wpdb;

        $tableName = $wpdb->prefix.'borlabs_cookie_blocked_content_types';

        $wpdb->query('
            DELETE FROM
                `'.$tableName.'`
            WHERE
                `type_id`="'.esc_sql($typeId).'"
        ');

        return true;
    }

    /**
     * getBlockedContentTypeDataByTypeId function.
     *
     * Get all information about a Blocked Content Type by its typeId
     *
     * @access public
     * @param mixed $typeId
     * @return void
     */
    public function getBlockedContentTypeDataByTypeId($typeId)
    {
        return ContentBlocker::getInstance()->getDataOfBlockedContentType($typeId);
    }

    /**
     * getBlockedContentTypeDataById function.
     *
     * Get all information about a Blocked Content Type by its id.
     * Use this function during the validation process when a Blocked Content Type is being edited and about to be saved.
     *
     * @access public
     * @param mixed $Id
     * @return void
     */
    public function getBlockedContentTypeDataById($id)
    {
        $bctData = BlockedContentTypes::getInstance()->get($id);

        return [
            'typeId'=>$bctData->type_id,
            'name'=>$bctData->name,
            'hosts'=>$bctData->hosts,
            'previewHTML'=>$bctData->preview_html,
            'globalJS'=>$bctData->global_js,
            'initJS'=>$bctData->init_js,
            'settings'=>$bctData->settings,
        ];
    }

    /**
     * blockContent function.
     *
     * Lets you block any content and returns the preview code for the Blocked Content Type.
     *
     * @access public
     * @param mixed $content: Your content you want to be blocked.
     * @param mixed $typeId: The Blocked Content Type typeId
     * @param string $title (default: ''): You can change the title for your blocked content
     * @return void
     */
    public function blockContent($content, $typeId, $title = '')
    {
        return ContentBlocker::getInstance()->handleContentBlocking($content, '', $typeId, $title);
    }

    /**
     * updateBlockedContent function.
     *
     * This function updates the blocked content of the $id. The $id is provided when using the filter "borlabsCookie/bct/modify_content/{typeId}"
     *
     * @access public
     * @param mixed $typeId
     * @param mixed $id
     * @param mixed $content
     * @return void
     */
    public function updateBlockedContent($typeId, $id, $content)
    {
        return JavaScript::getInstance()->addBlockedContent($typeId, $id, $content);
    }

    /**
     * updateBlockedContentTypeJavaScript function.
     *
     * This function lets you update the JavaScript and settings of your Blocked Content Type during the process when WordPress delivers a page.
     * This function does not update the JavaScript and settings of the Blocked Content Type in general!
     *
     * @access public
     * @param mixed $typeId
     * @param string $globalJS (default: '')
     * @param string $initJS (default: '')
     * @param mixed $settings (default: [])
     * @return void
     */
    public function updateBlockedContentTypeJavaScript($typeId, $globalJS = '', $initJS = '', $settings = []) {
        return JavaScript::getInstance()->addBlockedContentType($typeId, $globalJS, $initJS, $settings);
    }

    /**
     * getCurrentTitleOfBlockedContentType function.
     *
     * This function returns the title of the current blocked content.
     * It is only available and should only be used within the filter "borlabsCookie/bct/modify_content/{typeId}".
     *
     * @access public
     * @return void
     */
    public function getCurrentTitleOfBlockedContentType () {
        return ContentBlocker::getInstance()->getCurrentTitle();
    }
}
